rm(list = ls())
library(yarrr); library(MuMIn)

## Import and tidy metabolism data
met.data <- read.csv("trout_metabolism.csv")
met.data$resp <- met.data$resp / 1429   ## Divide by the density of O2 to convert milligrams (mass) to millilitres (volume) (1.429 g/l at STP)
met.data$resp <- met.data$resp * 20.1   ## Change that to energetic equivalents (J/h) using the Peters conversion (multiply by 20.1)
boltz <- 0.00008617343   ## The Boltzmann constant
T0 <- 273.15 + 10   ## Approximately the mean temperature of all the streams
met.data$temp.kT <- ((273.15+met.data$temp)-T0)/(boltz*(273.15+met.data$temp)*T0)
met.data$acclimation <- c("cold","warm","warm")[match(met.data$fish,c(12,1,5))]
met.data$acclimation3 <- c("cold","tepid","warm")[match(met.data$fish,c(12,1,5))]
m.met <- lm(log(resp) ~ log(mass) + temp.kT, data=met.data)

## Import and tidy feeding rate data
fr.data <- read.csv("trout_feeding.csv")
fr.data$temp.kT <- ((273.15+fr.data$temp)-T0)/(boltz*(273.15+fr.data$temp)*T0)
fr.data$acclimation <- c("cold","warm","warm")[match(fr.data$fish,c(12,1,5))]
fr.data$acclimation3 <- c("cold","tepid","warm")[match(fr.data$fish,c(12,1,5))]
rad <- fr.data[fr.data$prey=="radix",]
rad$calories <- 0.8 * 2.024 * 4.184 # Assuming 0.8 mg dry weight and caloric content of Gastropoda from Cummins & Wuycheck (1971)
rad$cons <- rad$feeding * rad$calories
m.fr.rad <- lm(log(cons+1) ~ temp.kT + acclimation + temp.kT:acclimation, data=rad)
m.fr.rad3 <- lm(log(cons+1) ~ temp.kT + acclimation3 + temp.kT:acclimation3, data=rad)
sim <- fr.data[fr.data$prey=="simuliidae",]
sim$calories <- 0.5 * 5.521 * 4.184 # Assuming 0.5 mg AFDM and caloric content of Simulium spp. from Cummins & Wuycheck (1971)
sim$cons <- sim$feeding * sim$calories
m.fr.sim <- lm(log(cons+1) ~ temp.kT + acclimation + temp.kT:acclimation, data=sim)
m.fr.sim3 <- lm(log(cons+1) ~ temp.kT + acclimation3 + temp.kT:acclimation3, data=sim)

## Calculate energetic efficiencies
rates <- data.frame(mass=met.data$mass, temp=met.data$temp, temp.kT=met.data$temp.kT, acclimation=met.data$acclimation, MR=met.data$resp)
## Calculate model-predicted feeding rates
rates$FR.rad[rates$acclimation=="cold"] <- exp(coef(m.fr.rad)[1] + coef(m.fr.rad)[2] * rates$temp.kT[rates$acclimation=="cold"])-1
rates$FR.rad[rates$acclimation=="warm"] <- exp(coef(m.fr.rad)[1] + coef(m.fr.rad)[3] + (coef(m.fr.rad)[2] + coef(m.fr.rad)[4]) * rates$temp.kT[rates$acclimation=="warm"])-1
rates$FR.sim[rates$acclimation=="cold"] <- exp(coef(m.fr.sim)[1] + coef(m.fr.sim)[2] * rates$temp.kT[rates$acclimation=="cold"])-1
rates$FR.sim[rates$acclimation=="warm"] <- exp(coef(m.fr.sim)[1] + coef(m.fr.sim)[3] + (coef(m.fr.sim)[2] + coef(m.fr.sim)[4]) * rates$temp.kT[rates$acclimation=="warm"])-1
## Assimilation efficiency for carnivores should be scaled with temperature according to Lang et al. (2017)
rates$temp.kT2 <- ((273.15+rates$temp)-293.15)/(boltz*(273.15+rates$temp)*293.15)
rates$AE <- exp(2.266)*exp(0.164*rates$temp.kT2) / (1 + exp(2.266)*exp(0.164*rates$temp.kT2))
## Calculate energetic efficiency as AE * FR / MR
rates$EE.rad <- rates$AE * rates$FR.rad / rates$MR
rates$EE.sim <- rates$AE * rates$FR.sim / rates$MR
rates$col <- c("blue","red")[match(rates$acclimation,c("cold","warm"))]
rates$pch <- c(16,17)[match(rates$acclimation,c("cold","warm"))]

m1.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT + log(mass):acclimation + temp.kT:acclimation + log(mass):temp.kT:acclimation, data=rates)
m2.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT + log(mass):acclimation + temp.kT:acclimation, data=rates)
m3.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT + log(mass):acclimation, data=rates)
m4.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT + temp.kT:acclimation, data=rates)
m5.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation + log(mass):acclimation + temp.kT:acclimation, data=rates)
m6.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT, data=rates)
m7.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + log(mass):temp.kT, data=rates)
m8.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation + log(mass):acclimation, data=rates)
m9.rad <- lm(log(EE.rad) ~ log(mass) + acclimation + log(mass):acclimation, data=rates)
m10.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation + temp.kT:acclimation, data=rates)
m11.rad <- lm(log(EE.rad) ~ temp.kT + acclimation + temp.kT:acclimation, data=rates)
m12.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation, data=rates)
m13.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT, data=rates)
m14.rad <- lm(log(EE.rad) ~ log(mass) + acclimation, data=rates)
m15.rad <- lm(log(EE.rad) ~ temp.kT + acclimation, data=rates)
m16.rad <- lm(log(EE.rad) ~ log(mass), data=rates)
m17.rad <- lm(log(EE.rad) ~ temp.kT, data=rates)
m18.rad <- lm(log(EE.rad) ~ acclimation, data=rates)
m19.rad <- lm(log(EE.rad) ~ 1, data=rates)
aic.rad <- AICc(m1.rad,m2.rad,m3.rad,m4.rad,m5.rad,m6.rad,m7.rad,m8.rad,m9.rad,m10.rad,m11.rad,m12.rad,m13.rad,m14.rad,m15.rad,m16.rad,m17.rad,m18.rad,m19.rad)
aic.rad$dAIC <- data.frame(aic.rad[2] - min(aic.rad[2]))$AIC; aic.rad
summary(m10.rad); anova(m10.rad)
new.rad <- data.frame(temp.kT=rep(seq(min(rates$temp.kT), max(rates$temp.kT), length.out=100),2), mass=mean(rates$mass), acclimation=c(rep("cold",100),rep("warm",100)))
p.rad <- data.frame(predict(m10.rad, newdata=new.rad, interval="confidence"))
p.rad.cold <- p.rad[1:100,]; p.rad.warm <- p.rad[101:200,]
boxplot(log(EE.rad) ~ acclimation, data=rates)

m1.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT + log(mass):acclimation + temp.kT:acclimation + log(mass):temp.kT:acclimation, data=rates)
m2.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT + log(mass):acclimation + temp.kT:acclimation, data=rates)
m3.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT + log(mass):acclimation, data=rates)
m4.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT + temp.kT:acclimation, data=rates)
m5.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation + log(mass):acclimation + temp.kT:acclimation, data=rates)
m6.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT, data=rates)
m7.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + log(mass):temp.kT, data=rates)
m8.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation + log(mass):acclimation, data=rates)
m9.sim <- lm(log(EE.sim) ~ log(mass) + acclimation + log(mass):acclimation, data=rates)
m10.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation + temp.kT:acclimation, data=rates)
m11.sim <- lm(log(EE.sim) ~ temp.kT + acclimation + temp.kT:acclimation, data=rates)
m12.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation, data=rates)
m13.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT, data=rates)
m14.sim <- lm(log(EE.sim) ~ log(mass) + acclimation, data=rates)
m15.sim <- lm(log(EE.sim) ~ temp.kT + acclimation, data=rates)
m16.sim <- lm(log(EE.sim) ~ log(mass), data=rates)
m17.sim <- lm(log(EE.sim) ~ temp.kT, data=rates)
m18.sim <- lm(log(EE.sim) ~ acclimation, data=rates)
m19.sim <- lm(log(EE.sim) ~ 1, data=rates)
aic.sim <- AICc(m1.sim,m2.sim,m3.sim,m4.sim,m5.sim,m6.sim,m7.sim,m8.sim,m9.sim,m10.sim,m11.sim,m12.sim,m13.sim,m14.sim,m15.sim,m16.sim,m17.sim,m18.sim,m19.sim)
aic.sim$dAIC <- data.frame(aic.sim[2] - min(aic.sim[2]))$AIC; aic.sim
summary(m10.sim); anova(m10.sim)
new.sim <- data.frame(temp.kT=rep(seq(min(rates$temp.kT), max(rates$temp.kT), length.out=100),2), mass=mean(rates$mass), acclimation=c(rep("cold",100),rep("warm",100)))
p.sim <- data.frame(predict(m10.sim, newdata=new.sim, interval="confidence"))
p.sim.cold <- p.sim[1:100,]; p.sim.warm <- p.sim[101:200,]
boxplot(log(EE.sim) ~ acclimation, data=rates)

## Saving statistical output and figures
write.table(as.matrix(c("Radix balthica", "", capture.output(aic.rad), capture.output(summary(m10.rad)), capture.output(anova(m10.rad)), "",
                        "Simulium vittatum", "", capture.output(aic.sim), capture.output(summary(m10.sim)), capture.output(anova(m10.sim)))),
            "Trout energetic efficiency.txt", row.names=F, col.names=F, quote=F)

tiff("Trout energetic efficiency.tif",16,8,pointsize=10,units="cm",res=600,compression="lzw")
par(mar=c(4.5,4.5,1,1), mfrow=c(1,2))
plot(log(EE.rad) ~ temp, pch=pch, col=transparent(col, trans.val=0.3), data=rates, xlab="", ylab="")
xval <- seq(min(rates$temp), max(rates$temp), length.out=100)
polygon(c(xval, rev(xval)), c(p.rad.cold$lwr, rev(p.rad.cold$upr)), col=adjustcolor("blue",alpha.f=0.25),  border = NA)
lines(xval, p.rad.cold$fit, col="blue", lwd=2, lty=1)
polygon(c(xval, rev(xval)), c(p.rad.warm$lwr, rev(p.rad.warm$upr)), col=adjustcolor("red",alpha.f=0.25),  border = NA)
lines(xval, p.rad.warm$fit, col="red", lwd=2, lty=2)
mtext(quote(Temperature~("C")), side=1, line=3)
mtext(quote(Ln~energetic~efficiency), side=2, line=3)
mtext("(a)", side=3, font=2, line=-1.2, adj=0.02)

plot(log(EE.sim) ~ temp, pch=pch, col=transparent(col, trans.val=0.3), data=rates, xlab="", ylab="", ylim=c(3,7))
polygon(c(xval, rev(xval)), c(p.sim.cold$lwr, rev(p.sim.cold$upr)), col=adjustcolor("blue",alpha.f=0.25),  border = NA)
lines(xval, p.sim.cold$fit, col="blue", lwd=2, lty=1)
polygon(c(xval, rev(xval)), c(p.sim.warm$lwr, rev(p.sim.warm$upr)), col=adjustcolor("red",alpha.f=0.25),  border = NA)
lines(xval, p.sim.warm$fit, col="red", lwd=2, lty=2)
mtext(quote(Temperature~("C")), side=1, line=3)
mtext(quote(Ln~energetic~efficiency), side=2, line=3)
mtext("(b)", side=3, font=2, line=-1.2, adj=0.02)
legend("topright", legend=c("cold (IS12)","warm (IS1+IS5)"), pch=16:17, col=c("blue","red"), lty=1:2, cex=0.9)
dev.off()


## Calculate energetic efficiencies with three levels of acclimation
rates <- data.frame(mass=met.data$mass, temp=met.data$temp, temp.kT=met.data$temp.kT, acclimation=met.data$acclimation3, MR=met.data$resp)
## Calculate model-predicted feeding rates
rates$FR.rad[rates$acclimation=="cold"] <- exp(coef(m.fr.rad3)[1] + coef(m.fr.rad3)[2]*rates$temp.kT[rates$acclimation=="cold"])-1
rates$FR.rad[rates$acclimation=="tepid"] <- exp(coef(m.fr.rad3)[1] + coef(m.fr.rad3)[3] + (coef(m.fr.rad3)[2] + coef(m.fr.rad3)[5]) * rates$temp.kT[rates$acclimation=="tepid"])-1
rates$FR.rad[rates$acclimation=="warm"] <- exp(coef(m.fr.rad3)[1] + coef(m.fr.rad3)[4] + (coef(m.fr.rad3)[2] + coef(m.fr.rad3)[6]) * rates$temp.kT[rates$acclimation=="warm"])-1
rates$FR.sim[rates$acclimation=="cold"] <- exp(coef(m.fr.sim3)[1] + coef(m.fr.sim3)[2]*rates$temp.kT[rates$acclimation=="cold"])-1
rates$FR.sim[rates$acclimation=="tepid"] <- exp(coef(m.fr.sim3)[1] + coef(m.fr.sim3)[3] + (coef(m.fr.sim3)[2] + coef(m.fr.sim3)[5]) * rates$temp.kT[rates$acclimation=="tepid"])-1
rates$FR.sim[rates$acclimation=="warm"] <- exp(coef(m.fr.sim3)[1] + coef(m.fr.sim3)[4] + (coef(m.fr.sim3)[2] + coef(m.fr.sim3)[6]) * rates$temp.kT[rates$acclimation=="warm"])-1
## Assimilation efficiency for carnivores should be scaled with temperature according to Lang et al. (2017)
rates$temp.kT2 <- ((273.15+rates$temp)-293.15)/(boltz*(273.15+rates$temp)*293.15)
rates$AE <- exp(2.266)*exp(0.164*rates$temp.kT2) / (1 + exp(2.266)*exp(0.164*rates$temp.kT2))
## Calculate energetic efficiency as AE * FR / MR
rates$EE.rad <- rates$AE * rates$FR.rad / rates$MR
rates$EE.sim <- rates$AE * rates$FR.sim / rates$MR
rates$col <- c("blue","orange","red")[match(rates$acclimation,c("cold","tepid","warm"))]
rates$pch <- c(16,17,18)[match(rates$acclimation,c("cold","tepid","warm"))]

m1.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT + log(mass):acclimation + temp.kT:acclimation + log(mass):temp.kT:acclimation, data=rates)
m2.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT + log(mass):acclimation + temp.kT:acclimation, data=rates)
m3.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT + log(mass):acclimation, data=rates)
m4.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT + temp.kT:acclimation, data=rates)
m5.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation + log(mass):acclimation + temp.kT:acclimation, data=rates)
m6.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT, data=rates)
m7.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + log(mass):temp.kT, data=rates)
m8.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation + log(mass):acclimation, data=rates)
m9.rad <- lm(log(EE.rad) ~ log(mass) + acclimation + log(mass):acclimation, data=rates)
m10.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation + temp.kT:acclimation, data=rates)
m11.rad <- lm(log(EE.rad) ~ temp.kT + acclimation + temp.kT:acclimation, data=rates)
m12.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT + acclimation, data=rates)
m13.rad <- lm(log(EE.rad) ~ log(mass) + temp.kT, data=rates)
m14.rad <- lm(log(EE.rad) ~ log(mass) + acclimation, data=rates)
m15.rad <- lm(log(EE.rad) ~ temp.kT + acclimation, data=rates)
m16.rad <- lm(log(EE.rad) ~ log(mass), data=rates)
m17.rad <- lm(log(EE.rad) ~ temp.kT, data=rates)
m18.rad <- lm(log(EE.rad) ~ acclimation, data=rates)
m19.rad <- lm(log(EE.rad) ~ 1, data=rates)
aic.rad <- AICc(m1.rad,m2.rad,m3.rad,m4.rad,m5.rad,m6.rad,m7.rad,m8.rad,m9.rad,m10.rad,m11.rad,m12.rad,m13.rad,m14.rad,m15.rad,m16.rad,m17.rad,m18.rad,m19.rad)
aic.rad$dAIC <- data.frame(aic.rad[2] - min(aic.rad[2]))$AIC; aic.rad
summary(m10.rad); anova(m10.rad)
new.rad <- data.frame(temp.kT=rep(seq(min(rates$temp.kT), max(rates$temp.kT), length.out=100),3), mass=mean(rates$mass), acclimation=c(rep("cold",100),rep("tepid",100),rep("warm",100)))
p.rad <- data.frame(predict(m10.rad, newdata=new.rad, interval="confidence"))
p.rad.cold <- p.rad[1:100,]; p.rad.tepid <- p.rad[101:200,]; p.rad.warm <- p.rad[201:300,]
boxplot(log(EE.rad) ~ acclimation, data=rates)

m1.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT + log(mass):acclimation + temp.kT:acclimation + log(mass):temp.kT:acclimation, data=rates)
m2.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT + log(mass):acclimation + temp.kT:acclimation, data=rates)
m3.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT + log(mass):acclimation, data=rates)
m4.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT + temp.kT:acclimation, data=rates)
m5.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation + log(mass):acclimation + temp.kT:acclimation, data=rates)
m6.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation + log(mass):temp.kT, data=rates)
m7.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + log(mass):temp.kT, data=rates)
m8.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation + log(mass):acclimation, data=rates)
m9.sim <- lm(log(EE.sim) ~ log(mass) + acclimation + log(mass):acclimation, data=rates)
m10.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation + temp.kT:acclimation, data=rates)
m11.sim <- lm(log(EE.sim) ~ temp.kT + acclimation + temp.kT:acclimation, data=rates)
m12.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT + acclimation, data=rates)
m13.sim <- lm(log(EE.sim) ~ log(mass) + temp.kT, data=rates)
m14.sim <- lm(log(EE.sim) ~ log(mass) + acclimation, data=rates)
m15.sim <- lm(log(EE.sim) ~ temp.kT + acclimation, data=rates)
m16.sim <- lm(log(EE.sim) ~ log(mass), data=rates)
m17.sim <- lm(log(EE.sim) ~ temp.kT, data=rates)
m18.sim <- lm(log(EE.sim) ~ acclimation, data=rates)
m19.sim <- lm(log(EE.sim) ~ 1, data=rates)
aic.sim <- AICc(m1.sim,m2.sim,m3.sim,m4.sim,m5.sim,m6.sim,m7.sim,m8.sim,m9.sim,m10.sim,m11.sim,m12.sim,m13.sim,m14.sim,m15.sim,m16.sim,m17.sim,m18.sim,m19.sim)
aic.sim$dAIC <- data.frame(aic.sim[2] - min(aic.sim[2]))$AIC; aic.sim
summary(m10.sim); anova(m10.sim)
new.sim <- data.frame(temp.kT=rep(seq(min(rates$temp.kT), max(rates$temp.kT), length.out=100),3), mass=mean(rates$mass), acclimation=c(rep("cold",100),rep("tepid",100),rep("warm",100)))
p.sim <- data.frame(predict(m10.sim, newdata=new.sim, interval="confidence"))
p.sim.cold <- p.sim[1:100,]; p.sim.tepid <- p.sim[101:200,]; p.sim.warm <- p.sim[201:300,]
boxplot(log(EE.sim) ~ acclimation, data=rates)

## Saving statistical output and figures
write.table(as.matrix(c("Radix balthica", "", capture.output(aic.rad), capture.output(summary(m10.rad)), capture.output(anova(m10.rad)), "",
                        "Simulium vittatum", "", capture.output(aic.sim), capture.output(summary(m10.sim)), capture.output(anova(m10.sim)))),
            "Trout energetic efficiency (3).txt", row.names=F, col.names=F, quote=F)

tiff("Trout energetic efficiency (3).tif",16,8,pointsize=10,units="cm",res=600,compression="lzw")
par(mar=c(4.5,4.5,1,1), mfrow=c(1,2))
plot(log(EE.rad) ~ temp, pch=pch, col=transparent(col, trans.val=0.3), data=rates, xlab="", ylab="")
xval <- seq(min(rates$temp), max(rates$temp), length.out=100)
polygon(c(xval, rev(xval)), c(p.rad.cold$lwr, rev(p.rad.cold$upr)), col=adjustcolor("blue",alpha.f=0.25),  border = NA)
lines(xval, p.rad.cold$fit, col="blue", lwd=2, lty=1)
polygon(c(xval, rev(xval)), c(p.rad.tepid$lwr, rev(p.rad.tepid$upr)), col=adjustcolor("orange",alpha.f=0.25),  border = NA)
lines(xval, p.rad.tepid$fit, col="orange", lwd=2, lty=2)
polygon(c(xval, rev(xval)), c(p.rad.warm$lwr, rev(p.rad.warm$upr)), col=adjustcolor("red",alpha.f=0.25),  border = NA)
lines(xval, p.rad.warm$fit, col="red", lwd=2, lty=2)
mtext(quote(Temperature~("C")), side=1, line=3)
mtext(quote(Ln~energetic~efficiency), side=2, line=3)
mtext("(a)", side=3, font=2, line=-1.2, adj=0.02)

plot(log(EE.sim) ~ temp, pch=pch, col=transparent(col, trans.val=0.3), data=rates, xlab="", ylab="", ylim=c(2.7,7.5))
polygon(c(xval, rev(xval)), c(p.sim.cold$lwr, rev(p.sim.cold$upr)), col=adjustcolor("blue",alpha.f=0.25),  border = NA)
lines(xval, p.sim.cold$fit, col="blue", lwd=2, lty=1)
polygon(c(xval, rev(xval)), c(p.sim.tepid$lwr, rev(p.sim.tepid$upr)), col=adjustcolor("orange",alpha.f=0.25),  border = NA)
lines(xval, p.sim.tepid$fit, col="orange", lwd=2, lty=2)
polygon(c(xval, rev(xval)), c(p.sim.warm$lwr, rev(p.sim.warm$upr)), col=adjustcolor("red",alpha.f=0.25),  border = NA)
lines(xval, p.sim.warm$fit, col="red", lwd=2, lty=2)
mtext(quote(Temperature~("C")), side=1, line=3)
mtext(quote(Ln~energetic~efficiency), side=2, line=3)
mtext("(b)", side=3, font=2, line=-1.2, adj=0.02)
legend("topright", legend=c("cold (IS12)","warm (IS1)","warm (IS5)"), pch=16:18, col=c("blue","orange","red"), lty=1:2, cex=0.8)
dev.off()
